import { App } from 'vue';
import { ComputedRef } from 'vue';
import type { DefineComponent } from 'vue';
import { Events } from 'vue';
import { Plugin as Plugin_2 } from 'vue';
import { Ref } from 'vue';
import type { VNode } from 'vue';

export declare function addExitAnimateToNode(item: ToastOptions, clasback?: (node: HTMLElement) => void): void;

export declare const enum AnimationStep {
    Enter = 0,
    Exit = 1
}

export declare function appendFromQueue(): void;

export declare const Bounce: CSSTransitionProps;

export declare type BuiltInIconProps = HTMLOrSVGElement & IconProps;

export declare function cacheRenderInstance(app: App<Element>, id: Id): void;

export declare function clearContainers(withExitAnimation?: boolean): void;

export declare type CloseBtnType = boolean | ((props: CloseButtonProps) => VNode) | DefineComponent<IconProps, {}, {}>;

export declare interface CloseButtonProps {
    closeToast: (e: MouseEvent) => void;
    type: ToastType;
    ariaLabel?: string;
    theme: ToastTheme;
}

export declare const containerInstances: Record<string, App<Element>>;

export declare type Content = string | VNode | ((props: ToastContentProps) => VNode) | DefineComponent<{}, {}, any>;

export declare interface CSSTransitionProps {
    /**
     * Css class to apply when toast enter
     */
    enter: string;
    /**
     * Css class to apply when toast leave
     */
    exit: string;
    /**
     * Append current toast position to the classname.
     * If multiple classes are provided, only the last one will get the position
     * For instance `myclass--top-center`...
     * `Default: false`
     */
    appendPosition?: boolean;
    /**
     * Collapse toast smoothly when exit animation end
     * `Default: true`
     */
    collapse?: boolean;
    /**
     * Collapse transition duration (auto disable when collapse: false)
     * `Default: 300`
     */
    collapseDuration?: number;
}

export declare type Data = Record<string, unknown>;

export declare function doAppend(content: Content, options?: ToastOptions<{}>): void;

declare type EventHandlers<E> = {
    [K in keyof E]?: E[K] extends Function ? E[K] : (payload: E[K]) => void;
};

export declare const Flip: CSSTransitionProps;

export declare function getAllToast(): ToastOptions<{}>[];

/**
 * Get the toast by id, given it's in the DOM, otherwise returns null
 */
export declare function getToast(toastId: Id): ToastOptions<{}> | undefined;

export declare const globalCache: {
    lastUrl: string;
};

export declare const globalOptions: {
    [key: string]: ToastContainerOptions;
};

declare function handlePromise<T = unknown>(promise: Promise<T> | (() => Promise<T>), { pending, error, success }: ToastPromiseParams<T>, options?: OmitLoadingOptsToastOption): Promise<T>;

/**
 * Used when providing custom icon
 */
export declare interface IconProps {
    theme: ToastTheme;
    type: ToastType;
    path?: string;
}

export declare type IconType = boolean | string | number | VNode | ((props: IconProps) => VNode) | DefineComponent<IconProps, {}, {}>;

export declare type Id = number | string;

export declare interface IToastContainers {
    [containerId: Id]: ToastOptions[];
}

declare type Nullable<T> = {
    [P in keyof T]: T[P] | null;
};

declare type OmitLoadingOptsToastOption = Omit<ToastOptions, 'isLoading' | 'draggable'>;

declare type OmitThemeToastOption = Omit<ToastOptions, 'theme'>;

declare type OmitTypeToastOption = Omit<ToastOptions, 'type'>;

/**
 * options for toast
 */
export declare interface Options {
    /**
     * use like
     * ```
     * toast.info("Hello World.\n I am <b>Tom</b>", { dangerouslyHTMLString: true });
     * ```
     *
     * @default false
     */
    dangerouslyHTMLString?: boolean;
    /**
     * Support right to left content
     * @default false
     */
    rtl?: boolean;
    /** Used to identify the ToastContainer when working with multiple container. Also used to set the id attribute */
    containerId?: Id;
    /**
     * One of top-right, top-center, top-left, bottom-right, bottom-center, bottom-left
     * @mark {@link ToastPosition}
     * @default 'top-right'
     */
    position?: ToastPosition;
    /**
     * Delay in ms to close the toast. If set to false, the notification needs to be closed manually
     * @default 5000
     */
    autoClose?: number | boolean;
    /**
     * Pass a custom close button.
     * To remove the close button pass `false`
     */
    closeButton?: CloseBtnType;
    /**
     * A reference to a valid react-transition-group/Transition component
     * @default 'bounce'
     */
    transition?: ToastTransition | CSSTransitionProps;
    /**
     * Display or not the progress bar below the toast(remaining time)
     * @default false
     */
    hideProgressBar?: boolean;
    /**
     * Keep the timer running or not on hover
     * @default true
     */
    pauseOnHover?: boolean;
    /**
     * Pause the timer when the window loses focus
     * @default true
     */
    pauseOnFocusLoss?: boolean;
    /**
     * Dismiss toast on click
     * @default true
     */
    closeOnClick?: boolean;
    /**
     * Add optional classes to the toast wrapper
     * @default -
     */
    toastClassName?: string;
    /**
     * Add optional classes to the TransitionGroup container
     * @default ''
     */
    bodyClassName?: string;
    /**
     * Add optional inline style to the container
     * @default {}
     */
    style?: Record<string, any>;
    /**
     * Add optional classes to the progress bar
     * @default -
     */
    progressClassName?: string;
    /**
     * Add optional inline style to the progress bar
     * @default {}
     */
    progressStyle?: Record<string, any>;
    /**
     * Define the ARIA role for the toasts
     * @default 'alert'
     */
    role?: string;
    /**
     * One of auto, light, dark, colored
     * @description `auto` means automatically detects system theme colors
     * @mark {@link ToastTheme}
     * @default 'auto'
     */
    theme?: ToastTheme;
    /**
     * Used to display a custom icon. Set it to `false` to prevent
     * the icons from being displayed
     * @default -
     */
    icon?: IconType;
}

declare interface OtherProps extends ToastOptions, ToastContainerOptions {
    toastRef: Ref<HTMLDivElement | undefined>;
    loading: ComputedRef<boolean>;
    /** on propgress end or cancel */
    done?: () => void;
}

export declare const queue: {
    items: {
        toastId: Id;
        containerId: Id;
        toastContent: Content;
        toastProps: {
            toastId?: Id | undefined;
            updateId?: Id | undefined;
            content?: Content | undefined;
            data?: {} | undefined;
            type?: ToastType | undefined;
            delay?: number | undefined;
            onOpen?: (<T = {}>(props: T) => void) | undefined;
            onClose?: (<T_1 = {}>(props: T_1) => void) | undefined;
            onClick?: ((event: MouseEvent) => void) | undefined;
            toastStyle?: Record<string, any> | undefined;
            progress?: number | undefined;
            isLoading?: boolean | undefined;
            dangerouslyHTMLString?: boolean | undefined;
            rtl?: boolean | undefined;
            containerId?: Id | undefined;
            position?: ToastPosition | undefined;
            autoClose?: number | boolean | undefined;
            closeButton?: CloseBtnType | undefined;
            transition?: ToastTransition | {
                enter: string;
                exit: string;
                appendPosition?: boolean | undefined;
                collapse?: boolean | undefined;
                collapseDuration?: number | undefined;
            } | undefined;
            hideProgressBar?: boolean | undefined;
            pauseOnHover?: boolean | undefined;
            pauseOnFocusLoss?: boolean | undefined;
            closeOnClick?: boolean | undefined;
            toastClassName?: string | undefined;
            bodyClassName?: string | undefined;
            style?: Record<string, any> | undefined;
            progressClassName?: string | undefined;
            progressStyle?: Record<string, any> | undefined;
            role?: string | undefined;
            theme?: ToastTheme | undefined;
            icon?: IconType | undefined;
            clearOnUrlChange?: boolean | undefined;
            multiple?: boolean | undefined;
            limit?: number | undefined;
            newestOnTop?: boolean | undefined;
            containerClassName?: string | undefined;
        };
    }[];
};

export declare function removeContainer(containerId: Id, withExitAnimation?: boolean): void;

export declare const Slide: CSSTransitionProps;

export declare interface Toast {
    content: Content;
    props: ToastOptions;
}

/** default toast */
export declare const toast: {
    (content: Content, options?: ToastOptions): Id;
    /** info toast */
    info(content: Content, options?: OmitTypeToastOption): Id;
    /** error toast */
    error(content: Content, options?: OmitTypeToastOption): Id;
    /** warning toast */
    warning(content: Content, options?: OmitTypeToastOption): Id;
    warn: (content: Content, options?: OmitTypeToastOption) => Id;
    /** success toast */
    success(content: Content, options?: OmitTypeToastOption): Id;
    /** loading toast */
    loading(content: Content, options?: OmitLoadingOptsToastOption): Id;
    /** dark toast */
    dark(content: Content, options?: OmitThemeToastOption): Id;
    /** remove a toast */
    remove(toastId?: Id): void;
    /** clear all toast */
    clearAll(containerId?: Id, withExitAnimation?: boolean): void;
    /**
     * return true if one container is displaying the toast
     */
    isActive(toastId: Id): boolean;
    update(toastId: Id, options?: UpdateOptions): void;
    /**
     * Used for controlled progress bar.
     */
    done(id: Id): void;
    promise: typeof handlePromise;
    POSITION: {
        TOP_LEFT: ToastPosition;
        TOP_RIGHT: ToastPosition;
        TOP_CENTER: ToastPosition;
        BOTTOM_LEFT: ToastPosition;
        BOTTOM_RIGHT: ToastPosition;
        BOTTOM_CENTER: ToastPosition;
    };
    THEME: {
        AUTO: ToastTheme;
        LIGHT: ToastTheme;
        DARK: ToastTheme;
        COLORED: ToastTheme;
    };
    TYPE: {
        INFO: ToastType;
        SUCCESS: ToastType;
        WARNING: ToastType;
        ERROR: ToastType;
        DEFAULT: ToastType;
    };
    TRANSITIONS: {
        FLIP: ToastTransition;
        SLIDE: ToastTransition;
        ZOOM: ToastTransition;
        BOUNCE: ToastTransition;
    };
};

export declare const ToastActions: {
    /**
     * add a toast
     * @param _ ..
     * @param opts toast props
     */
    add(_: Content, opts: ToastSetting): void;
    /**
     * remove a toast
     * @param id toastId
     */
    remove(id?: Id): void;
    /**
     * update the toast
     * @param opts toast props
     */
    update(opts?: UpdateOptions<unknown>): void;
    /**
     * clear all toasts in container.
     * @param containerId container id
     */
    clear(containerId?: Id, withExitAnimation?: boolean): void;
    dismissCallback(evt: AnimationEvent): void;
    dismiss(toastId?: Id): void;
};

/**
 * ClassName for the elements - can take a function to build a classname or a raw string that is cx'ed to defaults
 */
export declare type ToastClassName = ((context?: {
    type?: ToastType;
    defaultClassName?: string;
    position?: ToastPosition;
    rtl?: boolean;
}) => string) | string;

/**
 * options for app.use
 */
export declare interface ToastContainerOptions extends Options {
    /**
     * clear all toasts on url change
     * @default true
     */
    clearOnUrlChange?: boolean;
    /**
     * support multiple
     * @default true
     */
    multiple?: boolean;
    /**
     * Limit the number of toast displayed at the same time
     *
     * @default undefined
     *
     * @ignore  Auto disabled when `multiple: false`
     */
    limit?: number;
    /**
     * Display newest toast on top
     * @default false
     */
    newestOnTop?: boolean;
    /**
     * Add optional classes to the container
     * @default -
     */
    containerClassName?: string;
}

export declare const toastContainers: IToastContainers;

export declare type ToastContent<T = unknown> = string | VNode | ((props: ToastContentProps<T>) => string | VNode) | DefineComponent<{}, {}, any> | (() => string);

export declare interface ToastContentProps<Data = {}> {
    closeToast?: (e?: MouseEvent) => void;
    toastProps?: ToastOptions;
    data: Data;
}

export declare type ToastFunc = {
    (content: Content, options?: ToastOptions): void;
};

export declare const ToastifyContainer: (props: Record<string, any> & {}) => any;

export declare type ToastItemStatus = 'added' | 'removed' | 'updated';

export declare interface ToastOptions<Data = {}> extends Options {
    /**
     * Set a custom `toastId`
     */
    toastId?: Id;
    /**
     * Used during update
     */
    updateId?: Id;
    /** toast content */
    content?: Content;
    /**
     * any additional data you want to pass toast("content", { data: {key: value} })
     * @default {}
     */
    data?: Data;
    /**
     * One of info, success, warning, error, default, loading
     * @mark {@link ToastType}
     * @default 'default'
     */
    type?: ToastType;
    /**
     * Let you delay the toast appearance. Pass a value in ms
     * @default -
     */
    delay?: number;
    /**
     * Called when toast is mounted.
     */
    onOpen?: <T = {}>(props: T) => void;
    /**
     * Called when toast is unmounted.
     */
    onClose?: <T = {}>(props: T) => void;
    /**
     * Called when click inside Toast notification
     * @default -
     */
    onClick?: (event: MouseEvent) => void;
    /**
     * An optional inline style to apply.
     */
    toastStyle?: Record<string, any>;
    /**
     * Set the percentage for the controlled progress bar. `Value must be between 0 and 1.`
     */
    progress?: number;
    /** Only available when using `toast.loading' */
    isLoading?: boolean;
}

export declare type ToastPosition = 'top-left' | 'top-right' | 'top-center' | 'bottom-left' | 'bottom-right' | 'bottom-center';

export interface ToastPromiseParams<T = unknown> {
    pending?: string | UpdateOptions<void>;
    success?: string | UpdateOptions<T>;
    error?: string | UpdateOptions<any>;
}

declare type ToastSetting = ToastOptions & ToastContainerOptions;

export declare type ToastTheme = 'auto' | 'light' | 'dark' | 'colored';

export declare type ToastTransition = 'zoom' | 'flip' | 'bounce' | 'slide';

export declare type ToastType = 'info' | 'success' | 'error' | 'warning' | 'loading' | 'default';

export declare function updateGlobalOptions(options?: Partial<ToastContainerOptions>): void;

export declare interface UpdateOptions<T = unknown> extends Nullable<ToastOptions<T>> {
    /**
     * Used to update a toast.
     * Pass any valid ReactNode(string, number, component)
     */
    render?: ToastContent<T>;
}

export declare function useCssTransition(props: CSSTransitionProps & OtherProps): {
    isIn: Ref<boolean>;
    isRunning: Ref<boolean>;
    hideToast: (e?: MouseEvent) => void;
    eventHandlers: ComputedRef<EventHandlers<Events>>;
};

declare const Vue3Toastify: Plugin_2;
export default Vue3Toastify;

export declare const Zoom: CSSTransitionProps;

export { }
